<?php
// header.php
// Expect session_start() already called by including page.
?>

<!-- Header markup (drop into your header area) -->
<!DOCTYPE html>
<html>
<head>
  <title>Savithri Shop</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .sticky-header {
      position: sticky;
      top: 0;
      z-index: 1000;
      background: white;
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    /* Add this to your existing CSS */
.header-icon {
    position: relative;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    text-decoration: none;
    color: #333;
    width: 36px;
    height: 36px;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.cart-count {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #ff6b6b;
    color: white;
    width: 18px;
    height: 18px;
    border-radius: 50%;
    font-size: 0.6rem;
    font-weight: 700;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 2px solid white;
    display: none;
}

.cart-count.active {
    display: flex;
}
  </style>
</head>
<body>
<div class="sticky-header p-3 d-flex justify-content-between align-items-center">
  <div><img src="images/logo.jpeg" alt="Logo" height="40"></div>
  <div class="d-flex align-items-center">
   <?php if(isset($_SESSION['user_id'])): ?>
     <span class="me-3">Welcome, <?php echo htmlspecialchars($_SESSION['name']); ?></span>

    <?php endif; ?>
    <a href="cart.php" class="header-icon me-3">
      <i class="fas fa-shopping-cart"></i>
      <span class="cart-count" id="cartCount">0</span>
    </a>
    <a href="home.php">
      <img src="images/home-button.png" alt="Home" style="width:30px;height:30px;">
    </a>
  </div>
</div>
<!-- Live cart JS: fetches count, updates badge, listens for events -->
<script>
(function() {
  // expose userId for client-side decisions (set by PHP)
  const userId = <?= isset($_SESSION['user_id']) ? json_encode((int)$_SESSION['user_id']) : 'null' ?>;
  const CART_POLL_INTERVAL = 15000; // 15s polling for cross-device updates
  const cartCountEl = document.getElementById('cartCount');

  function setBadge(count) {
    if (!cartCountEl) return;
    if (!count || count <= 0) {
      cartCountEl.style.display = 'none';
      cartCountEl.textContent = '0';
      cartCountEl.classList.remove('active');
    } else {
      cartCountEl.style.display = 'flex';
      cartCountEl.textContent = count > 9 ? '9+' : String(count);
      cartCountEl.classList.add('active');
      // small attention animation
      try {
        cartCountEl.animate([{ transform: 'scale(1)' }, { transform: 'scale(1.2)' }, { transform: 'scale(1)' }], { duration: 260 });
      } catch (e) {}
    }
  }

  // fetch count from server
  async function fetchCount() {
    try {
      const res = await fetch('get_cart_count.php', { credentials: 'same-origin' });
      if (!res.ok) throw new Error('Network');
      const json = await res.json();
      if (json && typeof json.count !== 'undefined') {
        setBadge(Number(json.count || 0));
      }
    } catch (err) {
      // silent fail
      // console.warn('fetchCount error', err);
    }
  }

  // Update from event detail or numeric argument
  function updateFromEvent(value) {
    if (typeof value === 'number') {
      setBadge(value);
    } else if (value && typeof value.detail === 'object' && typeof value.detail.count !== 'undefined') {
      setBadge(Number(value.detail.count || 0));
    }
  }

  // Listen for custom event fired by pages after add_to_cart response
  document.addEventListener('cartUpdated', (e) => updateFromEvent(e));

  // BroadcastChannel for cross-tab update within same browser
  let bc;
  try {
    bc = new BroadcastChannel('cart_channel');
    bc.onmessage = (ev) => {
      if (ev && ev.data && typeof ev.data.count !== 'undefined') {
        setBadge(Number(ev.data.count || 0));
      }
    };
  } catch (e) {
    bc = null;
  }

  // storage event fallback (other tabs)
  window.addEventListener('storage', (e) => {
    if (e.key === 'cart_sync_count') {
      const newCount = Number(e.newValue || 0);
      setBadge(newCount);
    }
  });

  // initial load
  fetchCount();

  // periodic polling for cross-device updates (other devices cannot be notified instantly)
  setInterval(fetchCount, CART_POLL_INTERVAL);

  // helper that pages can call to update the header immediately:
  window.__updateCartBadge = function(count) {
    setBadge(Number(count || 0));
    // notify other tabs
    try {
      if (bc) bc.postMessage({ count: Number(count || 0) });
    } catch (e) {}
    try {
      localStorage.setItem('cart_sync_count', String(count || 0));
    } catch (e) {}
  };
})();
</script>
