<?php
session_start();
require 'config.php'; // Your database connection file

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$order_id = intval($_GET['order_id'] ?? 0);

// Get order details
$order_query = $conn->prepare("
    SELECT o.*, u.name as user_name, u.mobile, u.email 
    FROM orders o
    JOIN users u ON o.user_id = u.id
    WHERE o.id = ? AND o.user_id = ?
");
$order_query->bind_param('ii', $order_id, $_SESSION['user_id']);
$order_query->execute();
$order = $order_query->get_result()->fetch_assoc();

if (!$order) {
    die('Order not found or access denied');
}

// Get order items
$items_query = $conn->prepare("
    SELECT * FROM order_items 
    WHERE order_id = ?
");
$items_query->bind_param('i', $order_id);
$items_query->execute();
$items = $items_query->get_result()->fetch_all(MYSQLI_ASSOC);

// Calculate total
$subtotal = 0;
foreach ($items as $item) {
    $subtotal += $item['price'] * $item['quantity'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order Receipt #<?= $order['order_number'] ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        @media print {
            .no-print { display: none !important; }
            body { padding: 20px; }
        }
        .receipt { max-width: 800px; margin: 0 auto; }
        .logo { max-width: 200px; margin-bottom: 20px; }
        .total-row { font-size: 1.2em; font-weight: bold; }
    </style>
</head>
<body>
    <div class="container py-4 receipt">
        <div class="text-center mb-4">
            <img src="your-logo.png" alt="Logo" class="logo">
            <h2>Order Receipt</h2>
        </div>

        <div class="row mb-4">
            <div class="col-md-6">
                <h5>Order Details</h5>
                <p>
                    <strong>Order #:</strong> <?= htmlspecialchars($order['order_number']) ?><br>
                    <strong>Date:</strong> <?= date('F j, Y', strtotime($order['created_at'])) ?><br>
                    <strong>Status:</strong> <?= ucfirst($order['status']) ?>
                </p>
            </div>
            <div class="col-md-6 text-md-end">
                <h5>Customer Details</h5>
                <p>
                    <strong>Name:</strong> <?= htmlspecialchars($order['user_name']) ?><br>
                    <strong>Mobile:</strong> <?= htmlspecialchars($order['mobile']) ?><br>
                    <strong>Email:</strong> <?= htmlspecialchars($order['email'] ?? 'N/A') ?>
                </p>
            </div>
        </div>

        <div class="table-responsive mb-4">
            <table class="table table-bordered">
                <thead class="table-light">
                    <tr>
                        <th>Item</th>
                        <th>Type</th>
                        <th class="text-end">Price</th>
                        <th class="text-center">Qty</th>
                        <th class="text-end">Total</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $item): ?>
                    <tr>
                        <td><?= htmlspecialchars($item['product_name']) ?></td>
                        <td><?= ucfirst(htmlspecialchars($item['product_type'])) ?></td>
                        <td class="text-end">₹<?= number_format($item['price'], 2) ?></td>
                        <td class="text-center"><?= $item['quantity'] ?></td>
                        <td class="text-end">₹<?= number_format($item['price'] * $item['quantity'], 2) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
                <tfoot>
                    <tr class="table-light total-row">
                        <td colspan="4" class="text-end">Subtotal:</td>
                        <td class="text-end">₹<?= number_format($subtotal, 2) ?></td>
                    </tr>
                </tfoot>
            </table>
        </div>

        <div class="text-center mt-4 no-print">
            <button onclick="window.print()" class="btn btn-primary me-2">
                <i class="fas fa-print"></i> Print Receipt
            </button>
            <a href="my_orders.php" class="btn btn-outline-secondary">
                <i class="fas fa-arrow-left"></i> Back to Orders
            </a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://kit.fontawesome.com/your-code.js" crossorigin="anonymous"></script>
</body>
</html>